//this file is part of notepad++
//Copyright (C)2003 Don HO ( donho@altern.org )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <windows.h>
#include <ShellAPI.h>
#include "ScintillaEditView.h"
#include "Parameters.h"


// initialize the static variable
HINSTANCE ScintillaEditView::_hLib = ::LoadLibrary("SciLexer.DLL");
int ScintillaEditView::_refCount = 0;
UserDefineDialog ScintillaEditView::_userDefineDlg;

const int ScintillaEditView::_SC_MARGE_LINENUMBER = 0;
const int ScintillaEditView::_SC_MARGE_SYBOLE = 1;
const int ScintillaEditView::_SC_MARGE_FOLDER = 2;

const int ScintillaEditView::_MARGE_LINENUMBER_NB_CHIFFRE = 5;

/*
SC_MARKNUM_*     | Arrow               Plus/minus           Circle tree                 Box tree 
-------------------------------------------------------------------------------------------------------------
FOLDEROPEN       | SC_MARK_ARROWDOWN   SC_MARK_MINUS     SC_MARK_CIRCLEMINUS            SC_MARK_BOXMINUS 
FOLDER           | SC_MARK_ARROW       SC_MARK_PLUS      SC_MARK_CIRCLEPLUS             SC_MARK_BOXPLUS 
FOLDERSUB        | SC_MARK_EMPTY       SC_MARK_EMPTY     SC_MARK_VLINE                  SC_MARK_VLINE 
FOLDERTAIL       | SC_MARK_EMPTY       SC_MARK_EMPTY     SC_MARK_LCORNERCURVE           SC_MARK_LCORNER 
FOLDEREND        | SC_MARK_EMPTY       SC_MARK_EMPTY     SC_MARK_CIRCLEPLUSCONNECTED    SC_MARK_BOXPLUSCONNECTED 
FOLDEROPENMID    | SC_MARK_EMPTY       SC_MARK_EMPTY     SC_MARK_CIRCLEMINUSCONNECTED   SC_MARK_BOXMINUSCONNECTED 
FOLDERMIDTAIL    | SC_MARK_EMPTY       SC_MARK_EMPTY     SC_MARK_TCORNERCURVE           SC_MARK_TCORNER 
*/

const int ScintillaEditView::_markersArray[][NB_FOLDER_STATE] = {
  {SC_MARKNUM_FOLDEROPEN, SC_MARKNUM_FOLDER, SC_MARKNUM_FOLDERSUB, SC_MARKNUM_FOLDERTAIL, SC_MARKNUM_FOLDEREND, SC_MARKNUM_FOLDEROPENMID, SC_MARKNUM_FOLDERMIDTAIL},
  {SC_MARK_MINUS, SC_MARK_PLUS, SC_MARK_EMPTY, SC_MARK_EMPTY, SC_MARK_EMPTY, SC_MARK_EMPTY, SC_MARK_EMPTY},
  {SC_MARK_ARROWDOWN, SC_MARK_ARROW, SC_MARK_EMPTY, SC_MARK_EMPTY, SC_MARK_EMPTY, SC_MARK_EMPTY, SC_MARK_EMPTY},
  {SC_MARK_CIRCLEMINUS, SC_MARK_CIRCLEPLUS, SC_MARK_VLINE, SC_MARK_LCORNERCURVE, SC_MARK_CIRCLEPLUSCONNECTED, SC_MARK_CIRCLEMINUSCONNECTED, SC_MARK_TCORNERCURVE},
  {SC_MARK_BOXMINUS, SC_MARK_BOXPLUS, SC_MARK_VLINE, SC_MARK_LCORNER, SC_MARK_BOXPLUSCONNECTED, SC_MARK_BOXMINUSCONNECTED, SC_MARK_TCORNER}
};

//const int MASK_RED   = 0xFF0000;
//const int MASK_GREEN = 0x00FF00;
//const int MASK_BLUE  = 0x0000FF;

void ScintillaEditView::init(HINSTANCE hInst, HWND hPere)
{
	if (!_hLib)
	{
		MessageBox( NULL, "can't not load the dynamic library", "SYS ERR : ", MB_OK | MB_ICONSTOP);
		throw int(106901);
	}

	Window::init(hInst, hPere);
   _hSelf = ::CreateWindowEx(
					WS_EX_CLIENTEDGE,\
					"Scintilla",\
					"Notepad++",\
					WS_CHILD | WS_VSCROLL | WS_HSCROLL | WS_CLIPCHILDREN,\
					0, 0, 100, 100,\
					_hParent,\
					NULL,\
					_hInst,\
					NULL);

	if (!_hSelf)
	{
		systemMessage("System Err");
		throw int(106901);
	}

	_pScintillaFunc = (SCINTILLA_FUNC)::SendMessage(_hSelf, SCI_GETDIRECTFUNCTION, 0, 0);
	_pScintillaPtr = (SCINTILLA_PTR)::SendMessage(_hSelf, SCI_GETDIRECTPOINTER, 0, 0);

    _userDefineDlg.init(_hInst, _hParent, this);

	if (!_pScintillaFunc || !_pScintillaPtr)
	{
		systemMessage("System Err");
		throw int(106901);
	}

    execute(SCI_SETMARGINMASKN, _SC_MARGE_FOLDER, SC_MASK_FOLDERS);
    //execute(SCI_SETMARGINWIDTHN, _SC_MARGE_FOLDER, 16);
    showMargin(_SC_MARGE_FOLDER, true);

    execute(SCI_SETMARGINSENSITIVEN, _SC_MARGE_FOLDER, true);
    execute(SCI_SETMARGINSENSITIVEN, _SC_MARGE_SYBOLE, true);

    execute(SCI_SETPROPERTY, reinterpret_cast<WPARAM>("fold"), reinterpret_cast<LPARAM>("1"));
    //execute(SCI_SETPROPERTY, reinterpret_cast<WPARAM>("fold.compact"), reinterpret_cast<LPARAM>("1"));

	execute(SCI_SETPROPERTY, reinterpret_cast<WPARAM>("fold.html"), reinterpret_cast<LPARAM>("1"));
	execute(SCI_SETPROPERTY, reinterpret_cast<WPARAM>("fold.comment"), reinterpret_cast<LPARAM>("1"));
	execute(SCI_SETPROPERTY, reinterpret_cast<WPARAM>("fold.preprocessor"), reinterpret_cast<LPARAM>("1"));
    execute(SCI_SETFOLDFLAGS, 16, 0);

	for (int i = 0 ; i < NB_FOLDER_STATE ; i++)
        defineMarker(_markersArray[FOLDER_TYPE][i], _markersArray[_folderStyle][i], white, black);

	_codepage = ::GetACP();
    _pParameter = NppParameters::getInstance();

};

void ScintillaEditView::setStyle(int styleID, COLORREF fgColour, COLORREF bgColour, const char *fontName, int fontStyle, int fontSize)
{
    if (!((fgColour >> 24) & 0xFF))
	    execute(SCI_STYLESETFORE, styleID, fgColour);

    if (!((fgColour >> 24) & 0xFF))
	    execute(SCI_STYLESETBACK, styleID, bgColour);
    
    if ((!fontName)||(strcmp(fontName, "")))
		execute(SCI_STYLESETFONT, (WPARAM)styleID, (LPARAM)fontName);

    if ((fontStyle != -1) && (fontStyle != 0))
    {
        if (fontStyle & FONTSTYLE_BOLD)
            execute(SCI_STYLESETBOLD, (WPARAM)styleID, (LPARAM)true);
        if (fontStyle & FONTSTYLE_ITALIC)
            execute(SCI_STYLESETITALIC, (WPARAM)styleID, (LPARAM)true);
    }

	if (fontSize > 0)
		execute(SCI_STYLESETSIZE, styleID, fontSize);

}

void ScintillaEditView::setM30Lexer()
{/*
    std::string commandM30, macroM30, commandLabel, macroLabel;
    _pParameter->getWordList(commandM30, LANG_M30, LANG_INDEX_INSTR);
    _pParameter->getWordList(macroM30, LANG_M30, LANG_INDEX_INSTR2);
    _pParameter->getWordList(commandLabel, LANG_M30, LANG_INDEX_TYPE);
    _pParameter->getWordList(macroLabel, LANG_M30, LANG_INDEX_TYPE2);
	execute(SCI_SETLEXER, SCLEX_M30);

	// 0 => SCE_M30_CMD
	execute(SCI_SETKEYWORDS, 0, reinterpret_cast<LPARAM>(commandM30.c_str()));
	// 1 => SCE_M30_MACRO
	execute(SCI_SETKEYWORDS, 1, reinterpret_cast<LPARAM>(macroM30.c_str()));
	// 2 => SCE_M30_CMD_LABEL
	execute(SCI_SETKEYWORDS, 2, reinterpret_cast<LPARAM>(commandLabel.c_str()));
	// 3 => SCE_M30_MACRO_LABEL
	execute(SCI_SETKEYWORDS, 3, reinterpret_cast<LPARAM>(macroLabel.c_str()));
	
	setStyle(SCE_M30_CMD, blue);
	setStyle(SCE_M30_MACRO, orange);
	setStyle(SCE_M30_CMD_LABEL, red);
	setStyle(SCE_M30_MACRO_LABEL, red);

	setFont(SCE_M30_NUMBER, "Courier New");
	setStyle(SCE_M30_NUMBER, cyan); //red by default
	setStyle(SCE_M30_COMMENTLINE, darkGreen); // blue by default
*/}

void ScintillaEditView::setPcomLexer()
{/*
    std::string pcomNativeCmd, pcomExtCmd, pcomDefaultBuf;

    _pParameter->getWordList(pcomNativeCmd, LANG_PCOM, LANG_INDEX_INSTR);
    _pParameter->getWordList(pcomExtCmd, LANG_PCOM, LANG_INDEX_INSTR2);
    _pParameter->getWordList(pcomDefaultBuf, LANG_PCOM, LANG_INDEX_TYPE);
	execute(SCI_SETLEXER, SCLEX_PCOM);

	execute(SCI_SETKEYWORDS, 0, reinterpret_cast<LPARAM>(pcomNativeCmd.c_str()));
	execute(SCI_SETKEYWORDS, 1, reinterpret_cast<LPARAM>(pcomExtCmd.c_str()));
	execute(SCI_SETKEYWORDS, 2, reinterpret_cast<LPARAM>(pcomDefaultBuf.c_str()));
	execute(SCI_SETKEYWORDS, 3, reinterpret_cast<LPARAM>(""));

    setStyle(STYLE_DEFAULT, black, white);
	setFont(SCE_PCOM_NUMBER, "Courier New");
	setStyle(SCE_PCOM_NUMBER, red); //red by default
	setStyle(SCE_PCOM_COMMENTLINE, darkGreen); // blue by default
	
	setFont(SCE_PCOM_NATIVE_CMD, NULL, true);
	setStyle(SCE_PCOM_NATIVE_CMD, darkBlue);
	setStyle(SCE_PCOM_EXT_CMD, blue);

	setFont(SCE_PCOM_DEFAULT_BUF, NULL, true, true);
	setStyle(SCE_PCOM_DEFAULT_BUF, purple);

	setStyle(SCE_PCOM_MACRO, orange);
	setStyle(SCE_PCOM_NCFORMAT_MACRO, grey);
*/}

void ScintillaEditView::setXmlLexer(LangType type)
{

    execute(SCI_SETSTYLEBITS, 7, 0);

	if (type == L_XML)
	{
        execute(SCI_SETLEXER, SCLEX_HTML);
		for (int i = 0 ; i < 4 ; i++)
			execute(SCI_SETKEYWORDS, i, reinterpret_cast<LPARAM>(""));

        makeStyle("xml");
	}
	else if ((type == L_HTML) || (type == L_PHP) || (type == L_ASP))
	{
        execute(SCI_SETLEXER, SCLEX_XML);

        const char *htmlKeyWords =_pParameter->getWordList(LANG_HTML, LANG_INDEX_INSTR);
        execute(SCI_SETKEYWORDS, 0, reinterpret_cast<LPARAM>(htmlKeyWords?htmlKeyWords:""));

		makeStyle("html");

        setEmbeddedJSLexer();
        setPhpEmbeddedLexer();
		setEmbeddedAspLexer();
	}

}

void ScintillaEditView::setEmbeddedJSLexer()
{
    const char *JSKeyWords = _pParameter->getWordList(LANG_JS, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 1, reinterpret_cast<LPARAM>(JSKeyWords?JSKeyWords:""));

	makeStyle("javascript");
	execute(SCI_STYLESETEOLFILLED, SCE_HJ_DEFAULT, true);
	execute(SCI_STYLESETEOLFILLED, SCE_HJ_COMMENT, true);
	execute(SCI_STYLESETEOLFILLED, SCE_HJ_COMMENTDOC, true);
}

void ScintillaEditView::setPhpEmbeddedLexer()
{
    const char *PHPKeyWords = _pParameter->getWordList(LANG_JS, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 4, reinterpret_cast<LPARAM>(PHPKeyWords?PHPKeyWords:""));

	makeStyle("php");
	execute(SCI_STYLESETEOLFILLED, SCE_HPHP_DEFAULT, true);
	execute(SCI_STYLESETEOLFILLED, SCE_HPHP_COMMENT, true);
}

void ScintillaEditView::setEmbeddedAspLexer()
{
	const char *AspKeyWords = _pParameter->getWordList(LANG_VB, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 2, reinterpret_cast<LPARAM>(AspKeyWords?AspKeyWords:""));

    makeStyle("asp");
    execute(SCI_STYLESETEOLFILLED, SCE_HBA_DEFAULT, true);
}

void ScintillaEditView::setUserLexer()
{
    execute(SCI_SETLEXER, SCLEX_USER);

    execute(SCI_SETKEYWORDS, 0, reinterpret_cast<LPARAM>(_userDefineDlg.getBoolList()));
    for (int i = 0 ; i < _userDefineDlg.getNbWordList() ; i++)
    {
        const WordList & wl = _userDefineDlg.getWordList(i);
        execute(SCI_SETKEYWORDS, i + 1, reinterpret_cast<LPARAM>(wl._words));
        
        setStyle(SCE_USER_DEFINE_1+i, wl._color, RGB(0xFF, 0xFF, 0xFF), NULL, wl._fontStyle);
    }
    
    for (int i = 0 ; i < _userDefineDlg.getNbBlock() ; i++)
    {
	    execute(SCI_STYLESETBACK, SCE_USER_CROCHET + i, _userDefineDlg.getBlockColor(i));
    }
}

void ScintillaEditView::setIniLexer()
{
	execute(SCI_SETLEXER, SCLEX_PROPERTIES);
	execute(SCI_STYLESETEOLFILLED, SCE_PROPS_SECTION, true);
	makeStyle("ini");
}

void ScintillaEditView::setSqlLexer()
{
	execute(SCI_SETLEXER, SCLEX_SQL);

    const char *SQLKeyWords  = _pParameter->getWordList(LANG_SQL, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(SQLKeyWords?SQLKeyWords:""));
	makeStyle("sql");
}

void ScintillaEditView::setVBLexer()
{
    execute(SCI_SETLEXER, SCLEX_VB);

    const char *VBKeyWords  = _pParameter->getWordList(LANG_VB, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(VBKeyWords?VBKeyWords:""));
	makeStyle("vb");
}
void ScintillaEditView::setPascalLexer()
{
    execute(SCI_SETLEXER, SCLEX_PASCAL);

    const char *pascalKeyWords  = _pParameter->getWordList(LANG_PASCAL, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(pascalKeyWords?pascalKeyWords:""));
	makeStyle("pascal");
}

void ScintillaEditView::setPerlLexer()
{
    execute(SCI_SETLEXER, SCLEX_PERL);

    const char *perlKeyWords  = _pParameter->getWordList(LANG_PERL, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(perlKeyWords?perlKeyWords:""));
	makeStyle("perl");
}

void ScintillaEditView::setPythonLexer()
{
    execute(SCI_SETLEXER, SCLEX_PYTHON);

    const char *pythonKeyWords  = _pParameter->getWordList(LANG_PYTHON, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(pythonKeyWords?pythonKeyWords:""));
	makeStyle("python");
}

void ScintillaEditView::setCppLexer(LangType type)
{
    const char *cppInstrs;
    const char *cppTypes;
    const char *doxygenKeyWords  = _pParameter->getWordList(LANG_CPP, LANG_INDEX_TYPE2);

    int lang;

	switch (type)
	{
		case L_C:
		{
            lang = LANG_C;
			break;
		}

		case L_H:
		case L_CPP:
		{
            lang = LANG_CPP;
			break;
		}
		case L_JAVA:
		{
            lang = LANG_JAVA;
			break;
		}
		case L_JS:
		{
            lang = LANG_JS;
			break;
		}

		case L_RC:
		{
            lang = LANG_RC;
			break;
		}
		default:
			return;
	}
    cppInstrs = _pParameter->getWordList(lang, LANG_INDEX_INSTR);
    cppTypes = _pParameter->getWordList(lang, LANG_INDEX_TYPE);

    execute(SCI_SETLEXER, SCLEX_CPP); 
	if (isCJK())
	{
		int charSet = codepage2CharSet();
		if (charSet)
			execute(SCI_STYLESETCHARACTERSET, SCE_C_STRING, charSet);
	}
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(cppInstrs?cppInstrs:""));
    execute(SCI_SETKEYWORDS, 1, (LPARAM)(cppTypes?cppTypes:""));
	if (type != L_RC)
    {
        if (doxygenKeyWords)
            execute(SCI_SETKEYWORDS, 2, (LPARAM)doxygenKeyWords);
    }
	if (lang == LANG_JS)
	{
		LexerStyler *pStyler = (_pParameter->getLStylerArray()).getLexerStylerByName("javascript");	
		if (pStyler)
		{
			for (int i = 0 ; i < pStyler->getNbStyler() ; i++)
			{
				Style & style = pStyler->getStyler(i);
				int cppID = style._styleID; 
				switch (style._styleID)
				{
					case SCE_HJ_DEFAULT : cppID = SCE_C_DEFAULT; break;
					case SCE_HJ_WORD : cppID = SCE_C_IDENTIFIER; break;
					case SCE_HJ_SYMBOLS : cppID = SCE_C_OPERATOR; break;
					case SCE_HJ_COMMENT : cppID = SCE_C_COMMENT; break;
					case SCE_HJ_COMMENTLINE : cppID = SCE_C_COMMENTLINE; break;
					case SCE_HJ_COMMENTDOC : cppID = SCE_C_COMMENTDOC; break;
					case SCE_HJ_NUMBER : cppID = SCE_C_NUMBER; break;
					case SCE_HJ_KEYWORD : cppID = SCE_C_WORD; break;
					case SCE_HJ_DOUBLESTRING : cppID = SCE_C_STRING; break;
					case SCE_HJ_SINGLESTRING : cppID = SCE_C_CHARACTER; break;
					case SCE_HJ_REGEX : cppID = SCE_C_REGEX; break;
				}
				setStyle(cppID, style._fgColor, style._bgColor, style._fontName, style._fontStyle, style._fontSize);
			}
		}
		execute(SCI_STYLESETEOLFILLED, SCE_C_DEFAULT, true);
		execute(SCI_STYLESETEOLFILLED, SCE_C_COMMENTLINE, true);
		execute(SCI_STYLESETEOLFILLED, SCE_C_COMMENT, true);
		execute(SCI_STYLESETEOLFILLED, SCE_C_COMMENTDOC, true);
	}
	else
		makeStyle("cpp");
}


void ScintillaEditView::setObjCLexer() 
{
    execute(SCI_SETLEXER, SCLEX_CPP);
    const char *objcInstrs = _pParameter->getWordList(LANG_OBJC, LANG_INDEX_INSTR);
    const char *objcTypes = _pParameter->getWordList(LANG_OBJC, LANG_INDEX_TYPE);
	const char *doxygenKeyWords = _pParameter->getWordList(LANG_CPP, LANG_INDEX_TYPE2);
	const char *objCDirective = _pParameter->getWordList(LANG_OBJC, LANG_INDEX_INSTR2);
	const char *objCQualifier = _pParameter->getWordList(LANG_OBJC, LANG_INDEX_TYPE2);

    setStyle(SCE_OBJC_DIRECTIVE, blue, white);
	execute(SCI_SETKEYWORDS, 0, (LPARAM)(objcInstrs?objcInstrs:""));
    execute(SCI_SETKEYWORDS, 1, (LPARAM)(objcTypes?objcTypes:""));
	execute(SCI_SETKEYWORDS, 2, (LPARAM)(doxygenKeyWords?doxygenKeyWords:""));
	execute(SCI_SETKEYWORDS, 4, (LPARAM) (objCDirective?objCDirective:""));
	execute(SCI_SETKEYWORDS, 5, (LPARAM) (objCQualifier?objCQualifier:""));

	makeStyle("objc");
}

void ScintillaEditView::setMakefileLexer()
{
	execute(SCI_SETLEXER, SCLEX_MAKEFILE);
	makeStyle("makefile");
}

void ScintillaEditView::setCssLexer()
{
	execute(SCI_SETLEXER, SCLEX_CSS);

	const char *cssKeyWords  = _pParameter->getWordList(LANG_CSS, LANG_INDEX_INSTR);
    execute(SCI_SETKEYWORDS, 0, (LPARAM)(cssKeyWords?cssKeyWords:""));
	makeStyle("css");
}

void ScintillaEditView::defineDocType(LangType typeDoc)
{
	//setStyle(STYLE_DEFAULT, black, white, "Verdana", 0, 9);
    
    StyleArray & stylers = _pParameter->getMiscStylerArray();
    int iStyleDefault = stylers.getStylerIndexByID(STYLE_DEFAULT);
    if (iStyleDefault != -1)
    {
        Style & styleDefault = stylers.getStyler(iStyleDefault);
	    setStyle(styleDefault._styleID, styleDefault._fgColor, styleDefault._bgColor, styleDefault._fontName, styleDefault._fontStyle, styleDefault._fontSize);
    }

    execute(SCI_STYLECLEARALL);

    
    int caretWidth = 1;
    
	
    // surtout, surtout ne pas faire SCI_SETCODEPAGE ici
    // ca rame a mort!!!
	if (isCJK())
	{
		execute(SCI_SETCODEPAGE, _codepage);
	}

	execute(SCI_SETSTYLEBITS, 5);

	showMargin(_SC_MARGE_FOLDER, isNeededFolderMarge(typeDoc));
	switch (typeDoc)
	{
		case L_H :
		case L_C :
		case L_CPP :
		case L_JS:
		case L_JAVA :
		case L_RC :
            setCppLexer(typeDoc); break;

        case L_OBJC :
            setObjCLexer(); break;
		
	    case L_PHP :
		case L_ASP :
		case L_HTML :
		case L_XML :
			setXmlLexer(typeDoc); break;

		case L_CSS :
			setCssLexer(); break;

		case L_MAKEFILE :
			setMakefileLexer(); break;

		case L_INI :
			setIniLexer(); break;
			
		case L_M30 :
			setM30Lexer(); break;

		case L_PCOM :
			setPcomLexer(); break;

        case L_USER :
			setUserLexer(); break;

        case L_NFO :
			if (!_MSLineDrawFont)
				_MSLineDrawFont = ::AddFontResource(LINEDRAW_FONT);
			if (_MSLineDrawFont)
			{
				LexerStyler *pStyler = (_pParameter->getLStylerArray()).getLexerStylerByName("nfo");
				COLORREF bg = black;
				COLORREF fg = liteGrey;
				if (pStyler)
				{
                    
                    int i = pStyler->getStylerIndexByName("DEFAULT");
                    if (i != -1)
                    {
                        Style & style = pStyler->getStyler(i);
                        bg = style._bgColor;
                        fg = style._fgColor;
                    }
				}

				setStyle(STYLE_DEFAULT, fg, bg, "MS LineDraw");
				execute(SCI_STYLECLEARALL);
			}
			break;

		case L_SQL :
			setSqlLexer(); break;

		case L_VB :
			setVBLexer(); break;

		case L_PASCAL :
			setPascalLexer(); break;

		case L_PERL :
			setPerlLexer(); break;

		case L_PYTHON :
			setPythonLexer(); break;

		case L_TXT :
		default :
			execute(SCI_SETLEXER, (_codepage == CP_CHINESE_TRADITIONAL)?SCLEX_MAKEFILE:SCLEX_NULL); break;

	}
	//All the global styles should put here
	int index = stylers.getStylerIndexByID(STYLE_INDENTGUIDE);
	if (index != -1)
    {
        Style & style = stylers.getStyler(index);
	    setStyle(style._styleID, style._fgColor, style._bgColor, style._fontName, style._fontStyle, style._fontSize);
    }

	index = stylers.getStylerIndexByID(STYLE_BRACELIGHT);
	if (index != -1)
    {
        Style & style = stylers.getStyler(index);
	    setStyle(style._styleID, style._fgColor, style._bgColor, style._fontName, style._fontStyle, style._fontSize);
    }

	index = stylers.getStylerIndexByID(STYLE_BRACEBAD);
	if (index != -1)
    {
        Style & style = stylers.getStyler(index);
	    setStyle(style._styleID, style._fgColor, style._bgColor, style._fontName, style._fontStyle, style._fontSize);
    }

	execute(SCI_SETTABWIDTH, ((NppParameters::getInstance())->getNppGUI())._tabSize);
	execute(SCI_SETUSETABS, !((NppParameters::getInstance())->getNppGUI())._tabReplacedBySpace);

    execute(SCI_COLOURISE, 0, -1);
}

char * ScintillaEditView::attatchDefaultDoc(int nb)
{
	char title[10];
	char nb_str[4];
	strcat(strcpy(title, UNTITLED_STR), _itoa(nb, nb_str, 10));

	// get the doc pointer attached (by default) on the view Scintilla
	Document doc = execute(SCI_GETDOCPOINTER, 0, 0);

	// create the entry for our list
	_buffers.push_back(Buffer(doc, title));

	// set current index to 0
	_currentIndex = 0;

	return _buffers[_currentIndex]._fullPathName;
}


int ScintillaEditView::findDocIndexByName(const char *fn) const
{
	int index = -1;
	for (int i = 0 ; i < int(_buffers.size()) ; i++)
	{
		if (!strcmp(_buffers[i]._fullPathName, fn))
		{
			index = i;
			break;
		}
	}
	return index;
}

//! \brief this method activates the doc and the corresponding sub tab
//! \brief return the index of previeus current doc
char * ScintillaEditView::activateDocAt(int index)
{
	// before activating another document, we get the current position
	// from the Scintilla view then save it to the current document
	saveCurrentPos();
	Position & prevDocPos = _buffers[_currentIndex]._pos;

	// increase current doc ref count to 2 
	execute(SCI_ADDREFDOCUMENT, 0, _buffers[_currentIndex]._doc);

	// change the doc, this operation will decrease 
	// the ref count of old current doc to 1
	execute(SCI_SETDOCPOINTER, 0, _buffers[index]._doc);
	
	// reset all for another doc
	//execute(SCI_CLEARDOCUMENTSTYLE);
    //bool isDocTypeDiff = (_buffers[_currentIndex]._lang != _buffers[index]._lang);
	_currentIndex = index;
	
    //if (isDocTypeDiff)
    defineDocType(_buffers[_currentIndex]._lang);

	restoreCurrentPos(prevDocPos);

	execute(SCI_SETEOLMODE, _buffers[_currentIndex]._format);

    //execute(SCI_SETEOLMODE, _buffers[_currentIndex]._format);
	
    return _buffers[_currentIndex]._fullPathName;
}

// this method creates a new doc ,and adds it into 
// the end of the doc list and a last sub tab, then activate it
// it returns the name of this created doc (that's the current doc also)
char * ScintillaEditView::createNewDoc(const char *fn)
{
	Document newDoc = execute(SCI_CREATEDOCUMENT);
	_buffers.push_back(Buffer(newDoc, fn));
	_buffers[_buffers.size()-1].checkIfReadOnlyFile();
	return activateDocAt(int(_buffers.size())-1);
}

char * ScintillaEditView::createNewDoc(int nbNew)
{
	char title[10];
	char nb[4];
	strcat(strcpy(title, UNTITLED_STR), _itoa(nbNew, nb, 10));
	return createNewDoc(title);
}

// return the index to close then (argument) the index to activate
int ScintillaEditView::closeCurrentDoc(int & i2Activate)
{
	int oldCurrent = _currentIndex;

    Position & prevDocPos = _buffers[_currentIndex]._pos;

	// if the file 2 delete is the last one
	if (_currentIndex == int(_buffers.size()) - 1)
    {
		// if current index is 0, ie. the current is the only one
		if (!_currentIndex)
		{
			_currentIndex = 0;
		}
		// the current is NOT the only one and it is the last one,
		// we set it to the index which precedes it
		else
			_currentIndex -= 1;
    }
	// else the next current index will be the same,
	// we do nothing

	// get the iterator and calculate its position with the old current index value
	buf_vec_t::iterator posIt = _buffers.begin() + oldCurrent;

	// erase the position given document from our list
	_buffers.erase(posIt);

	// set another document, so the ref count of old active document owned
	// by Scintilla view will be decreased to 0 by SCI_SETDOCPOINTER message
	execute(SCI_SETDOCPOINTER, 0, _buffers[_currentIndex]._doc);

	defineDocType(_buffers[_currentIndex]._lang);
	restoreCurrentPos(prevDocPos);
	
    i2Activate = _currentIndex;
	
	return oldCurrent;
}

void ScintillaEditView::closeDocAt(int i2Close)
{
    execute(SCI_RELEASEDOCUMENT, 0, _buffers[i2Close]._doc);

	// get the iterator and calculate its position with the old current index value
	buf_vec_t::iterator posIt = _buffers.begin() + i2Close;

	// erase the position given document from our list
	_buffers.erase(posIt);

    _currentIndex -= (i2Close < _currentIndex)?1:0;
}

void ScintillaEditView::removeAllUnusedDocs()
{
	// unreference all docs  from list of Scintilla
	// by sending SCI_RELEASEDOCUMENT message
	for (int i = 0 ; i < int(_buffers.size()) ; i++)
		if (i != _currentIndex)
			execute(SCI_RELEASEDOCUMENT, 0, _buffers[i]._doc);
	
	// remove all docs except the current doc from list
	_buffers.clear();
}

void ScintillaEditView::getText(char *dest, int start, int end) 
{
	TextRange tr;
	tr.chrg.cpMin = start;
	tr.chrg.cpMax = end;
	tr.lpstrText = dest;
	execute(SCI_GETTEXTRANGE, 0, reinterpret_cast<LPARAM>(&tr));
}

void ScintillaEditView::marginClick(int position, int modifiers)
{
	int lineClick = int(execute(SCI_LINEFROMPOSITION, position, 0));
	int levelClick = int(execute(SCI_GETFOLDLEVEL, lineClick, 0));
	if (levelClick & SC_FOLDLEVELHEADERFLAG)
    {
		if (modifiers & SCMOD_SHIFT)
        {
			// Ensure all children visible
			execute(SCI_SETFOLDEXPANDED, lineClick, 1);
			expand(lineClick, true, true, 100, levelClick);
		}
        else if (modifiers & SCMOD_CTRL) 
        {
			if (execute(SCI_GETFOLDEXPANDED, lineClick, 0)) 
            {
				// Contract this line and all children
				execute(SCI_SETFOLDEXPANDED, lineClick, 0);
				expand(lineClick, false, true, 0, levelClick);
			} 
            else 
            {
				// Expand this line and all children
				execute(SCI_SETFOLDEXPANDED, lineClick, 1);
				expand(lineClick, true, true, 100, levelClick);
			}
		} 
        else 
        {
			// Toggle this line
			execute(SCI_TOGGLEFOLD, lineClick, 0);
		}
	}
}

void ScintillaEditView::expand(int &line, bool doExpand, bool force, int visLevels, int level)
{
	int lineMaxSubord = int(execute(SCI_GETLASTCHILD, line, level & SC_FOLDLEVELNUMBERMASK));
	line++;
	while (line <= lineMaxSubord)
    {
		if (force) 
        {
			if (visLevels > 0)
				execute(SCI_SHOWLINES, line, line);
			else
				execute(SCI_HIDELINES, line, line);
		} 
        else 
        {
			if (doExpand)
				execute(SCI_SHOWLINES, line, line);
		}
		int levelLine = level;
		if (levelLine == -1)
			levelLine = int(execute(SCI_GETFOLDLEVEL, line, 0));
		if (levelLine & SC_FOLDLEVELHEADERFLAG)
        {
			if (force) 
            {
				if (visLevels > 1)
					execute(SCI_SETFOLDEXPANDED, line, 1);
				else
					execute(SCI_SETFOLDEXPANDED, line, 0);
				expand(line, doExpand, force, visLevels - 1);
			} 
            else
            {
				if (doExpand)
                {
					if (!execute(SCI_GETFOLDEXPANDED, line, 0))
						execute(SCI_SETFOLDEXPANDED, line, 1);

					expand(line, true, force, visLevels - 1);
				} 
                else 
                {
					expand(line, false, force, visLevels - 1);
				}
			}
		}
        else
        {
			line++;
		}
	}
}

void ScintillaEditView::makeStyle(const char *lexerName)
{
	LexerStyler *pStyler = (_pParameter->getLStylerArray()).getLexerStylerByName(lexerName);
	if (pStyler)
	{
		for (int i = 0 ; i < pStyler->getNbStyler() ; i++)
		{
			Style & style = pStyler->getStyler(i);
			setStyle(style._styleID, style._fgColor, style._bgColor, style._fontName, style._fontStyle, style._fontSize);
		}
	}
}

void ScintillaEditView::performGlobalStyles() 
{
	StyleArray & stylers = _pParameter->getMiscStylerArray();

	int i = stylers.getStylerIndexByName("Current line background colour");
	if (i != -1)
	{
		Style & style = stylers.getStyler(i);
		execute(SCI_SETCARETLINEBACK, style._bgColor);
	}

	i = stylers.getStylerIndexByName("Mark colour");
	if (i != -1)
	{
		Style & style = stylers.getStyler(i);
		execute(SCI_MARKERSETFORE, 1, style._fgColor);
		execute(SCI_MARKERSETBACK, 1, style._bgColor);
	}

    COLORREF selectColorFore = yellow;
    COLORREF selectColorBack = grey;

	i = stylers.getStylerIndexByName("Selected text colour");
	if (i != -1)
    {
        Style & style = stylers.getStyler(i);
        selectColorFore = style._fgColor;
        selectColorBack = style._bgColor;
    }
    setSelectionColor(selectColorFore, selectColorBack);

    COLORREF caretColor = black;
	i = stylers.getStylerIndexByID(SCI_SETCARETFORE);
	if (i != -1)
    {
        Style & style = stylers.getStyler(i);
        caretColor = style._fgColor;
    }
    execute(SCI_SETCARETFORE, caretColor);
}

void ScintillaEditView::setLineIndent(int line, int indent) const {
	if (indent < 0)
		return;
	CharacterRange crange = getSelection();
	int posBefore = execute(SCI_GETLINEINDENTPOSITION, line);
	execute(SCI_SETLINEINDENTATION, line, indent);
	int posAfter = execute(SCI_GETLINEINDENTPOSITION, line);
	int posDifference = posAfter - posBefore;
	if (posAfter > posBefore) {
		// Move selection on
		if (crange.cpMin >= posBefore) {
			crange.cpMin += posDifference;
		}
		if (crange.cpMax >= posBefore) {
			crange.cpMax += posDifference;
		}
	} else if (posAfter < posBefore) {
		// Move selection back
		if (crange.cpMin >= posAfter) {
			if (crange.cpMin >= posBefore)
				crange.cpMin += posDifference;
			else
				crange.cpMin = posAfter;
		}
		if (crange.cpMax >= posAfter) {
			if (crange.cpMax >= posBefore)
				crange.cpMax += posDifference;
			else
				crange.cpMax = posAfter;
		}
	}
	execute(SCI_SETSEL, crange.cpMin, crange.cpMax);
}
