//this file is part of notepad++
//Copyright (C)2003 Don HO ( donho@altern.org )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef SHORTCUTS_H
#define SHORTCUTS_H

//#include "Parameters.h"
#include <vector>
#include <string>
#include <windows.h>
#include "shortcutRc.h"
#include "StaticDialog.h"

using namespace std;

const size_t nameLenMax = 64;

enum ShortcutType {TYPE_CMD, TYPE_MACRO, TYPE_USERCMD};

void getKeyStrFromVal(unsigned char keyVal, string & str);
ShortcutType getNameStrFromCmd(DWORD cmd, string & str);

struct Shortcut  : public StaticDialog {
	char _name[nameLenMax];
	bool _isCtrl;
	bool _isAlt;
	bool _isShift;
	unsigned char _key;
	bool _canModifyName;

	Shortcut():_isCtrl(false), _isAlt(false), _isShift(false), _key(0), _canModifyName(true) {_name[0] = '\0';};
	Shortcut(const Shortcut & shortcut) {
		this->_isCtrl = shortcut._isCtrl;
		this->_isAlt = shortcut._isAlt;
		this->_isShift = shortcut._isShift;
		this->_key = shortcut._key;
		strcpy(this->_name, shortcut._name);
	};
	Shortcut(const char *name, bool isCtrl, bool isAlt, bool isShift, unsigned char key) :\
		_isCtrl(isCtrl), _isAlt(isAlt), _isShift(isShift), _key(key){
		_name[0] = '\0';
		if (name)
			strcpy(_name, name);
	};

	int doDialog() {
		return ::DialogBoxParam(_hInst, MAKEINTRESOURCE(IDD_SHORTCUT_DLG), _hParent,  (DLGPROC)dlgProc, (LPARAM)this);
    };

	bool isValid() const {
		if (_key == 0)
			return false;
		// the following keys are always valid (NUMPAD & F1~F12 + ESC)
		if (((_key >= 0x60) && (_key <= 0x69)) || ((_key >= 0x70) && (_key <= 0x7B)) || (_key == 0x1B))
			return true;
		// the remain keys need at least Ctrl or Alt
		if ((_isCtrl) || (_isAlt))
			return true;
		return false;
	};
	void setNameReadOnly(bool canBeModified = false) {_canModifyName = canBeModified;};
	string toString();
	string toMenuItemString() {
		string str = _name;
		str += "\t";
		str += toString();
		return str;
	};
protected :
	BOOL CALLBACK run_dlgProc(UINT Message, WPARAM wParam, LPARAM lParam);

};
		 
class CommandShortcut : public Shortcut {
public:
	CommandShortcut(const char *name, long id, bool isCtrl, bool isAlt, bool isShift, unsigned char key) :\
		Shortcut(name, isCtrl, isAlt, isShift, key), _id() {};
	CommandShortcut(long id, Shortcut sc) :	Shortcut(sc), _id(id) {};
	long getID() const {return _id;};

private:
	long _id;
};


class Window;
class ScintillaEditView;

struct recordedMacroStep {
	enum MacroTypeIndex {mtUseLParameter, mtUseSParameter, mtMenuCommand, /*mtFunc*/};
	
	int message;
	long wParameter;
	long lParameter;
	string sParameter;
	MacroTypeIndex MacroType;
	
	recordedMacroStep(int iMessage, long wParam, long lParam);
	recordedMacroStep(int iCommandID) : message(0), wParameter(iCommandID), lParameter(0), MacroType(mtMenuCommand) {};

	recordedMacroStep(int type, int iMessage, long wParam, long lParam, const char *sParam)
		: message(iMessage), wParameter(wParam), lParameter(lParam), MacroType(MacroTypeIndex(type)){
		sParameter = *reinterpret_cast<const char *>(sParam);	
	};
	bool isValid(){
		return true;
	};

	void PlayBack(Window* pNotepad, ScintillaEditView *pEditView);
};

typedef vector<recordedMacroStep> Macro;

class MacroShortcut : public Shortcut {
friend class NppParameters;
public:
	MacroShortcut(Shortcut sc) : Shortcut(sc) {};
	MacroShortcut(Macro macro) : _macro(macro) {};
	MacroShortcut(Shortcut sc, Macro macro) : Shortcut(sc), _macro(macro) {};
	Macro & getMacro() {return _macro;};
private:
	Macro _macro;
};


class UserCommand : public Shortcut {
friend class NppParameters;
public:
	UserCommand(Shortcut sc) : Shortcut(sc) {};
	UserCommand(char *cmd) : _cmd(cmd) {};
	UserCommand(Shortcut sc, char *cmd) : Shortcut(sc), _cmd(cmd) {};
	const char* getCmd() const {return _cmd.c_str();};
private:
	string _cmd;
};

class Accelerator {
friend class ShortcutMapper;
public:
	Accelerator():_hAccTable(NULL), _didCopy(false), _pAccelArray(NULL), _nbAccelItems(0){
	};
	~Accelerator(){
		if (_didCopy)
			::DestroyAcceleratorTable(_hAccTable);
	};
	void init(HACCEL hAccel) {
		_hAccTable = hAccel;
		_nbOriginalAccelItem = ::CopyAcceleratorTable(_hAccTable, NULL, 0);
	};
	HACCEL getAccTable() const {return _hAccTable;};

	void uptdateShortcuts();

	void coloneAccelTable() {
		copyAccelArray();
		reNew();
	};


private:
	HACCEL _hAccTable;
	bool _didCopy;

	ACCEL *_pAccelArray;
	int _nbOriginalAccelItem;
	int _nbAccelItems;

	
	size_t copyAccelArray(int nbMacro2add = 0, int nbUserCmd2add = 0) {
		int newSize = _nbOriginalAccelItem + nbMacro2add + nbUserCmd2add;
		_nbAccelItems = newSize;

		if (_pAccelArray)
			delete [] _pAccelArray;
		_pAccelArray = new ACCEL[_nbAccelItems];

		::CopyAcceleratorTable(_hAccTable, _pAccelArray, _nbOriginalAccelItem);
		return newSize;
	};

	void reNew() {
		if (!_didCopy)
			_didCopy = true;
		else
			::DestroyAcceleratorTable(_hAccTable);

		_hAccTable = ::CreateAcceleratorTable(_pAccelArray, _nbAccelItems);
	};
};

#endif //SHORTCUTS_H
